// SPDX-License-Identifier: GPL-2.0-only
/*
 * usb devicetree helper functions
 */

#include <common.h>
#include <linux/usb/usb.h>
#include <linux/usb/phy.h>
#include <linux/usb/of.h>
#include <of.h>

static const char *usb_dr_modes[] = {
	[USB_DR_MODE_UNKNOWN]		= "",
	[USB_DR_MODE_HOST]		= "host",
	[USB_DR_MODE_PERIPHERAL]	= "peripheral",
	[USB_DR_MODE_OTG]		= "otg",
};

/**
 * usb_get_dr_mode_from_string() - Get dual role mode for given string
 * @str: String to find the corresponding dual role mode for
 *
 * This function performs a lookup for the given string and returns the
 * corresponding enum usb_dr_mode. If no match for the string could be found,
 * 'USB_DR_MODE_UNKNOWN' is returned.
 */
static enum usb_dr_mode usb_get_dr_mode_from_string(const char *str)
{
	int ret;

	ret = match_string(usb_dr_modes, ARRAY_SIZE(usb_dr_modes), str);
	return (ret < 0) ? USB_DR_MODE_UNKNOWN : ret;
}

/**
 * usb_get_dr_mode - Get dual role mode for given device
 * @dev:	Pointer to the given device
 *
 * The function gets phy interface string from property 'barebox,dr_mode'
 * or 'dr_mode' in the given device's device tree node and returns the
 * correspondig enum usb_dr_mode
 */
enum usb_dr_mode usb_get_dr_mode(struct device *dev)
{
	struct device_node *np = dev_of_node(dev);
	const char *dr_mode;
	int err;

	err = of_property_read_string(np, "barebox,dr_mode", &dr_mode);
	if (err < 0)
		err = of_property_read_string(np, "dr_mode", &dr_mode);
	if (err < 0)
		return USB_DR_MODE_UNKNOWN;

	return usb_get_dr_mode_from_string(dr_mode);
}
EXPORT_SYMBOL_GPL(usb_get_dr_mode);

static const char *usbphy_modes[] = {
	[USBPHY_INTERFACE_MODE_UNKNOWN]	= "",
	[USBPHY_INTERFACE_MODE_UTMI]	= "utmi",
	[USBPHY_INTERFACE_MODE_UTMIW]	= "utmi_wide",
	[USBPHY_INTERFACE_MODE_ULPI]	= "ulpi",
	[USBPHY_INTERFACE_MODE_SERIAL]	= "serial",
	[USBPHY_INTERFACE_MODE_HSIC]	= "hsic",
};

/**
 * of_usb_get_phy_mode - Get phy mode for given device_node
 * @np:	Pointer to the given device_node
 *
 * The function gets phy interface string from property 'phy_type',
 * and returns the correspondig enum usb_phy_interface
 */
enum usb_phy_interface of_usb_get_phy_mode(struct device_node *np,
		const char *propname)
{
	const char *phy_type;
	int err, i;

	if (!propname)
		propname = "phy_type";

	err = of_property_read_string(np, propname, &phy_type);
	if (err < 0)
		return USBPHY_INTERFACE_MODE_UNKNOWN;

	for (i = 0; i < ARRAY_SIZE(usbphy_modes); i++)
		if (!strcmp(phy_type, usbphy_modes[i]))
			return i;

	return USBPHY_INTERFACE_MODE_UNKNOWN;
}
EXPORT_SYMBOL_GPL(of_usb_get_phy_mode);

/**
 * of_usb_get_maximum_speed - Get maximum speed for given device_node
 * @np:	Pointer to the given device_node
 *
 * The function gets maximum speed string from property 'maximum-speed',
 * and returns the correspondig enum usb_device_speed
 */
enum usb_device_speed of_usb_get_maximum_speed(struct device_node *np,
		const char *propname)
{
	const char *maximum_speed;
	int err;

	if (!propname)
		propname = "maximum-speed";

	err = of_property_read_string(np, propname, &maximum_speed);
	if (err < 0)
		return USB_SPEED_UNKNOWN;

	return usb_speed_by_string(maximum_speed);
}
EXPORT_SYMBOL_GPL(of_usb_get_maximum_speed);
